#!/usr/bin/ruby
# vim: set sw=4 sts=4 et tw=80 :
# $Id$

%w{Paludis getoptlong}.each {|x| require x}

include Paludis

$0 = File.basename $0
Log.instance.log_level = LogLevel::Warning
Log.instance.program_name = $0

EQUAL = 0
PACKAGE = 1
TILDE = 2

class PackageID
    def my_string(style, show_repo)
        repo = show_repo ? "::#{repository_name}" : ''
        slot = slot_key ? ":#{slot_key.value}" : ''
        case style
        when EQUAL
            "=#{name}-#{version}#{slot}#{repo}"
        when TILDE
            "~#{name}-#{version.remove_revision}#{slot}#{repo}"
        when PACKAGE
            "#{name}#{repo}"
        end
    end
end

def write_file(arr, file, base)
    output = base
    arr.each {|x| output += "\n#{x}"}
    if @subdir
        dir = "#{file}.d"
        file = "#{dir}/#{@spec.to_s.gsub('/','_')}.conf"
    end

    if @pretend
        puts "\n# #{file}"
        puts output
    else
        if (@subdir && !File.directory?("#{@config_dir}/#{dir}"))
            puts "#{dir} is not a directory. Either specify \"--use-subdir no\" or create #{dir}"
            exit 1
        else
            File.open("#{@config_dir}/#{file}",'a') {|file| file.puts output}
        end
    end
end

def collect_highest_versions(dl)
    seen = {}
    dl.each do |entry|
        next unless entry.kind == DepListEntryKind::Masked
        pid = entry.package_id
        seen[pid.name] ||= {}
        seen[pid.name][:version] ||= VersionSpec.new('0')
        if pid.version > seen[pid.name][:version]
            seen[pid.name][:version] = pid.version
            seen[pid.name][:dle] = entry
        end
    end

    out = []
    seen.each_value {|v| out << v[:dle]}
    return out
end

opts = GetoptLong.new(
    [ '--help',                   '-h',  GetoptLong::NO_ARGUMENT ],
    [ '--version',                '-V',  GetoptLong::NO_ARGUMENT ],
    [ '--log-level',                     GetoptLong::REQUIRED_ARGUMENT ],
    [ '--environment',            '-E',  GetoptLong::REQUIRED_ARGUMENT ],
    [ '--match-type',             '-m',  GetoptLong::REQUIRED_ARGUMENT ],
    [ '--pretend',                '-p',  GetoptLong::NO_ARGUMENT ],
    [ '--override-masks',         '-o',  GetoptLong::REQUIRED_ARGUMENT ],
    [ '--use-subdir',             '-s',  GetoptLong::REQUIRED_ARGUMENT ],
    [ '--include-repository-name',       GetoptLong::REQUIRED_ARGUMENT ])

envspec = ''
@match_type = TILDE
@pretend = false
@repository = false
@subdir = false
@masks = nil;

dlomf = DepListOverrideMasksFunctions.new

opts.each do | opt, arg |
    case opt
    when '--help'
        puts "Usage: " + $0 + " [options] target"
        puts
        puts "Options:"
        puts "  --help                     Display a help message"
        puts "  --version                  Display libpaludis version"
        puts
        puts "  --pretend                  Display changes instead of changing files"
        puts "  --log-level level          Set log level (debug, qa, warning, silent)"
        puts "  --environment env          Environment specification (class:suffix, both parts optional, class must be 'paludis' if specified)" if @gt020
        puts "  --match-type type          Set match type (equal, tilde (default), package)"
        puts "  --include-repository-name  Include repository name in DepSpec (yes (default), no)"
        puts "  --override-masks list      List of masks to override, seperated by \",\", default is all"
        puts "                                license"
        puts "                                tilde_keywords"
        puts "                                unkeyworded"
        puts "                                repository_masks"
        puts "  --use-subdir               Write files to conf.d dirs (yes, no (default))"
        exit 0

    when '--version'
        puts "#{$0} (libpaludis version: #{Paludis::Version})"
        exit 0

    when '--log-level'
        case arg
        when 'debug'
            Log.instance.log_level = LogLevel::Debug
        when 'qa'
            Log.instance.log_level = LogLevel::Qa
        when 'warning'
            Log.instance.log_level = LogLevel::Warning
        when 'silent'
            Log.instance.log_level = LogLevel::Silent
        else
            $stderr.puts "Bad --log-level value " + arg
            exit 1
        end

    when '--environment'
            envspec = arg

    when '--match-type'
        case arg
        when 'package'
            @match_type = PACKAGE
        when 'equal'
            @match_type = EQUAL
        when 'tilde'
            @match_type = TILDE
        else
            $stderr.puts "Bad --match-type value #{arg}"
            exit 1
        end

    when '--include-repository-name'
        case arg
        when 'yes'
            @repository = true
        when 'no'
            @repository = false
        else
            $stderr.puts "Bad --include-repository-name value #{arg}"
            exit 1
        end

    when '--override-masks'
        @masks = arg

    when '--use-subdir'
        case arg
        when 'yes'
            @subdir = true
        when 'no'
            @subdir = false
        else
            $stderr.puts "Bad --use-subdir value #{arg}"
            exit 1
        end

    when '--pretend'
        @pretend = true

    end
end

if ARGV.length != 1
    puts "Please specify exactly 1 target"
    exit 1
end

target = ARGV.first

env = Paludis::EnvironmentFactory.instance.create(envspec)
if env.format_key.value != "paludis" then
    $stderr.puts "#$0: --environment must specify class 'paludis'"
    exit 1
end
@config_dir = env.config_location_key.value

if @masks.nil?
    dlomf.bind(:tilde_keywords, env)
    dlomf.bind(:unkeyworded, env)
    dlomf.bind(:repository_masks)
    dlomf.bind(:license)
else
    @masks.scan(/\w+/) do |mask|
        mask_sym = mask.to_sym
        case mask_sym
        when :tilde_keywords, :unkeyworded
            dlomf.bind(mask_sym, env)
        when :repository_masks, :license
            dlomf.bind(mask_sym)
        else
            $stderr.puts "Unrecognised --override-masks option: #{mask}"
            exit 1
        end
    end
end


db = env.package_database

@spec = nil

begin
    @spec = parse_user_package_dep_spec(target, env, [:throw_if_set],
                Filter::SupportsAction.new(InstallAction))
rescue GotASetNotAPackageDepSpec
    @spec = env.set(target)
end

dl = DepList.new(env, DepListOptions.new)
dl.options.override_masks = dlomf

begin
    dl.add(@spec, env.default_destinations)
rescue AllMaskedError
    bad_spec = $!.query
    pids = env[Selection::AllVersionsSorted.new(
                   Generator::Matches.new(parse_user_package_dep_spec(bad_spec, env, []),[]) |
                       Filter::SupportsAction.new(InstallAction))]
    if pids.empty?
        puts "No versions of #{bad_spec} are available."
    else
        puts "I'm not allowed to unmask #{bad_spec}: Masks are:"
        pids.each do |pid|
            print "#{pid}: Masked by: "
            reasons = []
            pid.masks.each do |reason|
                if reason.instance_of? UnacceptedMask
                    if reason.unaccepted_key.instance_of? MetadataKeywordNameSetKey
                        reasons << 'keyword'
                    end
                elsif reason.instance_of? RepositoryMask
                    reasons << 'repository'
                end
            end
            puts reasons.join(', ')
        end
    end
    exit 1
end

keywords = []
unmask = []
#license = []
collect_highest_versions(dl).each do |entry|
    reasons = entry.package_id.masks
    repo = db.fetch_repository(entry.package_id.repository_name)
    e_spec = entry.package_id.my_string(@match_type, @repository)
    reasons.each do |reason|
        if reason.instance_of? UnacceptedMask
            key = reason.unaccepted_key
            if key.instance_of? MetadataKeywordNameSetKey
                my_arch = repo.profile_variable("ARCH")
                kw = reason.unaccepted_key.value
                if kw.empty?
                    my_arch = '*'
                elsif kw.include?("~#{my_arch}")
                    my_arch = "~#{my_arch}"
                else
                    my_arch = kw.first
                    my_arch += (my_arch[0] == ?~ ? " #{my_arch[1..-1]}" : '')
                end
                keywords <<(e_spec + " #{my_arch}")
            end
        elsif reason.instance_of? RepositoryMask
            unmask << e_spec
        end
    end
end

output_base = "\n# Added by #{$0} for #{target}"
output_base += "\n" + ('#' * (output_base.length-1))

write_file(keywords, 'keywords.conf', output_base) unless keywords.empty?
write_file(unmask, 'package_unmask.conf', output_base) unless unmask.empty?
#write_file(license, 'license.conf', output_base) unless license.empty?



